package pf::SNMP::Meru;

=head1 NAME

pf::SNMP::Meru

=head1 SYNOPSIS

Module to manage Meru controllers

=cut

use strict;
use warnings;
use diagnostics;

use Log::Log4perl;
use Net::Appliance::Session;
use POSIX;

use base ('pf::SNMP');

use pf::config;
# importing switch constants
use pf::SNMP::constants;
use pf::util;

=head1 STATUS

Tested against MeruOS version 3.6.1-67

=head1 BUGS AND LIMITATIONS

=over

=item CLI deauthentication

De-authentication of a Wireless user is based on CLI access (Telnet or SSH).
This is a vendor issue and it might be fixed in newer firmware versions.

=item Per SSID VLAN Assignment on unencrypted network not supported

The vendor doesn't include the SSID in their RADIUS-Request when on MAC Authentication.
VLAN assignment per SSID is not possible.
This is a vendor issue and might be fixed in newer firmware versions.

=back
 
=head1 SUBROUTINES

=over

=cut

# CAPABILITIES
# access technology supported
sub supportsWirelessDot1x { return $TRUE; }
sub supportsWirelessMacAuth { return $TRUE; }

=item getVersion

obtain image version information from switch

=cut
sub getVersion {
    my ($this) = @_;
    my $oid_mwWncVarsSoftwareVersion = '1.3.6.1.4.1.15983.1.1.4.1.1.27'; # from meru-wlan
    my $logger = Log::Log4perl::get_logger( ref($this) );
    if ( !$this->connectRead() ) {
        return '';
    }

    # mwWncVarsSoftwareVersion sample output:
    # 3.6.1-67

    # first trying with a .0
    $logger->trace("SNMP get_request for mwWncVarsSoftwareVersion: $oid_mwWncVarsSoftwareVersion.0");
    my $result = $this->{_sessionRead}->get_request( -varbindlist => [$oid_mwWncVarsSoftwareVersion.".0"] );
    if (defined($result)) {
        return $result->{$oid_mwWncVarsSoftwareVersion.".0"};
    }

    # then trying straight
    $logger->trace("SNMP get_request for mwWncVarsSoftwareVersion: $oid_mwWncVarsSoftwareVersion");
    $result = $this->{_sessionRead}->get_request( -varbindlist => [$oid_mwWncVarsSoftwareVersion] );
    if (defined($result)) {
        return $result->{$oid_mwWncVarsSoftwareVersion};
    }

    # none of the above worked
    $logger->warn("unable to fetch version information");
}

=item parseTrap

This is called when we receive an SNMP-Trap for this device

=cut
sub parseTrap {
    my ( $this, $trapString ) = @_;
    my $trapHashRef;
    my $logger = Log::Log4perl::get_logger( ref($this) );

    $logger->debug("trap currently not handled");
    $trapHashRef->{'trapType'} = 'unknown';

    return $trapHashRef;
}

=item deauthenticateMac

deauthenticate a MAC address from wireless network

Right now te only way to do it is from the CLi (through Telnet or SSH).

=cut
sub deauthenticateMac {
    my ( $this, $mac ) = @_;
    my $logger = Log::Log4perl::get_logger( ref($this) );

    if ( !$this->isProductionMode() ) {
        $logger->info("not in production mode ... we won't deauthenticate $mac");
        return 1;
    }

    if ( length($mac) != 17 ) {
        $logger->error("MAC format is incorrect ($mac). Should be xx:xx:xx:xx:xx:xx");
        return 1;
    }

    my $session;
    eval {
        $session = Net::Appliance::Session->new(
            Host      => $this->{_ip},
            Timeout   => 5,
            Transport => $this->{_cliTransport},
            Platform => 'MeruOS',
            Source   => $lib_dir.'/pf/SNMP/Meru/nas-pb.yml'
        );
        $session->connect(
            Name     => $this->{_cliUser},
            Password => $this->{_cliPwd}
        );
    };

    if ($@) {
        $logger->error("Unable to connect to ".$this->{'_ip'}." using ".$this->{_cliTransport}.". Failed with $@");
        return;
    }

    #if (!$session->in_privileged_mode()) {
    #    if (!$session->enable($this->{_cliEnablePwd})) {
    #        $logger->error("Cannot get into privileged mode on ".$this->{'ip'}.
    #                       ". Are you sure you provided enable password in configuration?");
    #        $session->close();
    #        return;
    #    }
    #}

    # if $session->begin_configure() does not work, use the following command:
    # my $command = "configure terminal\nno station $mac\n";
    my $command = "no station $mac";

    $logger->info("Deauthenticating mac $mac");
    $logger->trace("sending CLI command '$command'");
    my @output;
    $session->in_privileged_mode(1);
    eval { 
        $session->begin_configure();
        @output = $session->cmd(String => $command, Timeout => '10');
    };
    $session->in_privileged_mode(0);
    if ($@) {
        $logger->error("Unable to deauthenticate $mac: $@");        
        $session->close();
        return;
    }
    $session->close();
    return 1;
}

=back

=head1 AUTHOR

Olivier Bilodeau <obilodeau@inverse.ca>

Francois Gaudreault <fgaudreault@inverse.ca>

Regis Balzard <rbalzard@inverse.ca>

=head1 COPYRIGHT

Copyright (C) 2010,2011 Inverse inc.

=head1 LICENSE

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
USA.

=cut

1;
